<?php

namespace App\Http\Controllers;

use App\Models\Item;
use Illuminate\Http\Request;

class ItemController extends Controller
{
    // Untuk Admin: CRUD Barang
    public function index()
    {
        $items = Item::all();
        return view('admin.items.index', compact('items'));
    }

    public function create()
    {
        return view('admin.items.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|unique:items,code|max:255',
            'description' => 'nullable|string',
            'quantity' => 'required|integer|min:0',
        ]);

        Item::create($request->all());

        return redirect()->route('admin.items.index')->with('success', 'Barang berhasil ditambahkan!');
    }

    public function show(Item $item)
    {
        return view('admin.items.show', compact('item'));
    }

    public function edit(Item $item)
    {
        return view('admin.items.edit', compact('item'));
    }

    public function update(Request $request, Item $item)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|unique:items,code,' . $item->id . '|max:255',
            'description' => 'nullable|string',
            'quantity' => 'required|integer|min:0',
        ]);

        $item->update($request->all());

        return redirect()->route('admin.items.index')->with('success', 'Barang berhasil diperbarui!');
    }

    public function destroy(Item $item)
    {
        // Tambahkan validasi agar barang tidak bisa dihapus jika sedang dipinjam
        if ($item->borrowings()->whereIn('status', ['pending', 'approved'])->exists()) {
            return redirect()->back()->with('error', 'Barang tidak bisa dihapus karena masih ada peminjaman yang aktif.');
        }

        $item->delete();
        return redirect()->route('admin.items.index')->with('success', 'Barang berhasil dihapus!');
    }

    // Untuk User: Melihat Daftar Barang yang Tersedia
    public function indexUser()
    {
        // Ambil item terdaftar yang tersedia dan juga item placeholder "Barang Lain"
        $items = Item::where('status', 'available')
                    ->where('quantity', '>', 0)
                    ->orWhere('code', 'BARANG-LAIN') // Sertakan item placeholder
                    ->orderBy('name')
                    ->get();
        return view('user.items.index', compact('items'));
    }

    /**
     * Mengembalikan stok barang yang tersedia untuk ID item tertentu sebagai JSON.
     */
    public function getStockQuantity(Request $request, Item $item)
    {
        // Jika item adalah placeholder "Barang Lain", kembalikan stok sangat besar atau "N/A"
        if ($item->code == 'BARANG-LAIN') {
            return response()->json(['quantity' => 'N/A', 'is_placeholder' => true]);
        }

        return response()->json(['quantity' => $item->quantity, 'is_placeholder' => false]);
    }
}