<?php

namespace App\Http\Controllers; // Pastikan ini sudah benar

use App\Models\Borrowing;
use App\Models\Item;
use Illuminate\Http\Request;
use Carbon\Carbon;

class BorrowingController extends Controller
{
    // ADMIN FUNCTIONS

    public function indexAdmin()
    {
        $borrowings = Borrowing::with(['user', 'item'])->latest()->get();
        return view('admin.borrowings.index', compact('borrowings'));
    }

    public function approve(Borrowing $borrowing)
    {
        // Logika persetujuan standar (tanpa penanganan 'Barang Lain')
        if ($borrowing->status !== 'pending') {
            return redirect()->back()->with('error', 'Peminjaman sudah tidak dalam status pending.');
        }

        if (!is_numeric($borrowing->quantity) || !is_numeric($borrowing->item->quantity) || (int)$borrowing->quantity > (int)$borrowing->item->quantity) {
            return redirect()->back()->with('error', 'Kuantitas peminjaman tidak valid atau melebihi stok yang tersedia. Stok tersedia: ' . ($borrowing->item->quantity ?? 'N/A'));
        }
        $borrowing->item->decrement('quantity', (int)$borrowing->quantity);

        $borrowing->update(['status' => 'approved']);
        return redirect()->route('admin.borrowings.index')->with('success', 'Peminjaman berhasil disetujui!');
    }

    public function reject(Borrowing $borrowing)
    {
        if ($borrowing->status !== 'pending') {
            return redirect()->back()->with('error', 'Peminjaman sudah tidak dalam status pending.');
        }

        $borrowing->update(['status' => 'rejected']);
        return redirect()->route('admin.borrowings.index')->with('success', 'Peminjaman berhasil ditolak!');
    }

    public function markAsReturned(Borrowing $borrowing)
    {
        // Logika pengembalian standar (tanpa penanganan 'Barang Lain')
        if ($borrowing->status === 'approved' || $borrowing->status === 'overdue') {
            if (!is_numeric($borrowing->quantity)) {
                return redirect()->back()->with('error', 'Kuantitas peminjaman tidak valid untuk dikembalikan.');
            }
            $borrowing->item->increment('quantity', (int)$borrowing->quantity);

            $borrowing->update([
                'status' => 'returned',
                'return_date' => Carbon::now()->toDateString(),
                'return_time' => Carbon::now()->toTimeString(),
            ]);

            return redirect()->route('admin.borrowings.index')->with('success', 'Barang berhasil dikembalikan dan stok diperbarui!');
        }
        return redirect()->back()->with('error', 'Status peminjaman tidak valid untuk dikembalikan (hanya "approved" atau "overdue" yang bisa).');
    }

    // USER FUNCTIONS
    public function create()
    {
        // Hanya tampilkan barang yang tersedia
        $items = Item::where('status', 'available')->where('quantity', '>', 0)->orderBy('name')->get();
        return view('user.borrowings.create', compact('items'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'item_id' => 'required|exists:items,id',
            'quantity' => 'required|integer|min:1',
            'borrow_date' => 'required|date|after_or_equal:today',
            'notes' => 'nullable|string|max:500',
        ]);

        $item = Item::find($request->item_id);

        if (!$item || $item->status !== 'available' || (int)$request->quantity > (int)$item->quantity) {
            return redirect()->back()->withInput()->with('error', 'Kuantitas yang diminta melebihi stok yang tersedia, atau barang tidak tersedia/valid.');
        }

        Borrowing::create([
            'user_id' => auth()->id(),
            'item_id' => $request->item_id,
            'quantity' => $request->quantity,
            'borrow_date' => $request->borrow_date,
            'borrow_time' => Carbon::now()->toTimeString(),
            'status' => 'pending',
            'notes' => $request->notes,
        ]);

        // Pengalihan kembali ke form dengan pesan sukses (bukan halaman detail)
        return redirect()->back()->with('success', 'Permintaan peminjaman berhasil diajukan. Menunggu persetujuan admin.');
    }

    public function indexUser()
    {
        $myBorrowings = Borrowing::where('user_id', auth()->id())->with('item')->latest()->get();
        return view('user.borrowings.index', compact('myBorrowings'));
    }

    // showSuccessDetail DIHAPUS
    // public function showSuccessDetail(Borrowing $borrowing) { ... }
}