<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Inventory;
use Illuminate\Http\Request;
use App\Imports\InventoriesImport; // BARIS INI DITAMBAHKAN
use Maatwebsite\Excel\Facades\Excel; // BARIS INI DITAMBAHKAN
use Maatwebsite\Excel\Validators\ValidationException; // BARIS INI DITAMBAHKAN

class InventoryController extends Controller
{
    /**
     * Menampilkan daftar semua item inventaris.
     */
    public function index()
    {
        $inventories = Inventory::latest()->get();
        return view('admin.inventories.index', compact('inventories'));
    }

    /**
     * Menampilkan form untuk membuat item inventaris baru.
     */
    public function create()
    {
        return view('admin.inventories.create');
    }

    /**
     * Menyimpan item inventaris yang baru dibuat ke database.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|unique:inventories,code|max:255',
            'serial_number' => 'nullable|string|unique:inventories,serial_number|max:255',
            'description' => 'nullable|string',
            'location' => 'nullable|string|max:255',
            'condition' => 'required|string|max:255',
            'purchase_date' => 'nullable|date',
            'owner' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        Inventory::create($request->all());

        return redirect()->route('admin.inventories.index')->with('success', 'Item inventaris berhasil ditambahkan!');
    }

    /**
     * Menampilkan detail item inventaris tertentu.
     */
    public function show(Inventory $inventory)
    {
        return view('admin.inventories.show', compact('inventory'));
    }

    /**
     * Menampilkan form untuk mengedit item inventaris tertentu.
     */
    public function edit(Inventory $inventory)
    {
        return view('admin.inventories.edit', compact('inventory'));
    }

    /**
     * Memperbarui item inventaris tertentu di database.
     */
    public function update(Request $request, Inventory $inventory)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|unique:inventories,code,' . $inventory->id . '|max:255',
            'serial_number' => 'nullable|string|unique:inventories,serial_number,' . $inventory->id . '|max:255',
            'description' => 'nullable|string',
            'location' => 'nullable|string|max:255',
            'condition' => 'required|string|max:255',
            'purchase_date' => 'nullable|date',
            'owner' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $inventory->update($request->all());

        return redirect()->route('admin.inventories.index')->with('success', 'Item inventaris berhasil diperbarui!');
    }

    /**
     * Menghapus item inventaris tertentu dari database.
     */
    public function destroy(Inventory $inventory)
    {
        $inventory->delete();
        return redirect()->route('admin.inventories.index')->with('success', 'Item inventaris berhasil dihapus!');
    }

    /**
     * Memproses impor file Excel inventaris.
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv', // Hanya izinkan Excel dan CSV
        ], [
            'file.required' => 'File Excel wajib diunggah.',
            'file.mimes' => 'Format file tidak valid. Harap unggah file .xlsx, .xls, atau .csv.',
        ]);

        try {
            // Impor data dari file
            Excel::import(new InventoriesImport, $request->file('file'));
            return redirect()->route('admin.inventories.index')->with('success', 'Data inventaris berhasil diimpor!');
        } catch (ValidationException $e) {
            // Tangani error validasi dari impor
            $failures = $e->failures();
            $errorMessage = "Gagal mengimpor data karena kesalahan validasi: <br>";
            foreach ($failures as $failure) {
                $errorMessage .= "Baris " . $failure->row() . ": " . implode(", ", $failure->errors()) . "<br>";
            }
            return redirect()->back()->with('error', $errorMessage);
        } catch (\Exception $e) {
            // Tangani error umum lainnya
            return redirect()->back()->with('error', 'Terjadi kesalahan saat mengimpor data: ' . $e->getMessage());
        }
    }
}