<?php

namespace App\Exports;

use App\Models\Borrowing;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize; // Opsional: untuk lebar kolom otomatis
use Carbon\Carbon; // Untuk format tanggal

class DailyReportExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize
{
    protected $today; // Properti untuk menyimpan tanggal laporan

    public function __construct(Carbon $today)
    {
        $this->today = $today;
    }

    /**
    * @return \Illuminate\Support\Collection
    */
    public function collection()
    {
        // Mengambil semua data peminjaman yang relevan untuk laporan harian
        // Anda bisa menyesuaikan query ini sesuai kebutuhan laporan Anda
        return Borrowing::with(['user', 'item'])
                        ->where(function ($query) {
                            $query->whereDate('created_at', $this->today) // Pengajuan baru dibuat hari ini
                                  ->orWhereDate('borrow_date', $this->today) // Barang yang dipinjam (disetujui) hari ini
                                  ->orWhereDate('return_date', $this->today); // Barang yang dikembalikan hari ini
                        })
                        ->latest()
                        ->get();
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'ID Peminjaman',
            'Nama Peminjam',
            'Keterangan Peminjam (Kelas/Email)', // Diperbarui untuk lebih jelas
            'Nama Barang',
            'Kode Barang',
            'Jumlah',
            'Tanggal Peminjaman',
            'Jam Peminjaman',
            'Tanggal Pengembalian',
            'Jam Pengembalian',
            'Catatan',
            'Status',
            'Waktu Dibuat',
            'Terakhir Diperbarui',
        ];
    }

    /**
     * @param mixed $borrowing
     * @return array
     */
    public function map($borrowing): array
    {
        // Menentukan nama dan keterangan peminjam
        $borrowerName = $borrowing->user ? $borrowing->user->name : $borrowing->guest_name;
        $borrowerDetail = $borrowing->user ? $borrowing->user->email : ($borrowing->guest_class ? 'Tamu - ' . $borrowing->guest_class : 'Tamu'); // Lebih fleksibel untuk tamu

        return [
            $borrowing->id,
            $borrowerName,
            $borrowerDetail,
            $borrowing->item->name,
            $borrowing->item->code,
            (int)$borrowing->quantity, // Pastikan sebagai integer
            Carbon::parse($borrowing->borrow_date)->format('d M Y'),
            Carbon::parse($borrowing->borrow_time)->format('H:i'),
            $borrowing->return_date ? Carbon::parse($borrowing->return_date)->format('d M Y') : '-',
            $borrowing->return_time ? Carbon::parse($borrowing->return_time)->format('H:i') : '-',
            $borrowing->notes ?? '-',
            ucfirst($borrowing->status),
            Carbon::parse($borrowing->created_at)->format('d M Y H:i:s'),
            Carbon::parse($borrowing->updated_at)->format('d M Y H:i:s'),
        ];
    }
}